from typing import Any
import argparse
import sys
import webbrowser
from pathlib import Path
from importlib import metadata

import logging
logger = logging.getLogger('millconnect.gui')

from PyQt6.QtCore import QTimer ,QRunnable, QThreadPool, pyqtSlot, pyqtSignal, QTranslator, QLocale
from PyQt6.QtWidgets import QApplication, QLabel, QWidget, QVBoxLayout, QProgressBar, QStackedWidget, QFileDialog
from PyQt6.QtGui import QIcon

from .project import Project
from .updater import newVersionAvailable, update
from .server import listen, handleConnection

class ProjectSelector(QWidget):
    folderSelected = pyqtSignal(Path)
    canceledSelection = pyqtSignal()

    def __init__(self, parent=None):
        super().__init__(parent)

        self.projectFolder: Path = None

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.label = QLabel(self, text=self.tr('Select a project folder.'))
        layout.addWidget(self.label)

    def showEvent(self, a0):
        super().showEvent(a0)
        QTimer.singleShot(0, self.selectFolder)

    def selectFolder(self):
        self.projectFolder = QFileDialog.getExistingDirectory(self, self.tr('Open project folder'))
        logger.debug(f'selected folder: "{self.projectFolder}".')

        if not self.projectFolder:
            self.canceledSelection.emit()
        else:
            self.folderSelected.emit(Path(self.projectFolder))

class UpdateRunner(QRunnable):

    def __init__(self, parent=None):
        super().__init__()
        self.parent = parent

    def run(self):
        self.checkForUpdate()

    def checkForUpdate(self):
        logger.info('Checking for update.')
        if newVersionAvailable():
            logger.info('New version available.')
            self.parent.downloadInitialiated()
            update()
            self.parent.restartSignal.emit()
        else:
            logger.info('No new version available.')
            self.parent.noUpdate()
            self.parent.noUpdateSignal.emit()

class UpdatingDialog(QWidget):
    noUpdateSignal = pyqtSignal()
    restartSignal = pyqtSignal()

    def __init__(self, parent=None, *args, **kwargs):
        super().__init__(parent, *args, **kwargs)

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.label = QLabel(self, text=self.tr('Checking for updates.'))
        layout.addWidget(self.label)

        self.progressBar = QProgressBar(self, minimum=0, maximum=100, value=0)
        layout.addWidget(self.progressBar)


    @pyqtSlot()
    def initializeUpdate(self):
        self.threadpool = QThreadPool()
        self.threadpool.start(UpdateRunner(self))

    @pyqtSlot()
    def downloadInitialiated(self):
        self.label.setText(self.tr('Downloading...'))
        self.progressBar.setValue(50)
        self.update()

    @pyqtSlot()
    def restartInitiated(self):
        self.label.setText(self.tr('Update successfull. Restarting...'))
        self.progressBar.setValue(100)
        self.update()

    @pyqtSlot()
    def noUpdate(self):
        self.label.setText(self.tr('Application is up-to-date.'))
        self.progressBar.setValue(100)
        self.update()

class WebsocketServer(QRunnable):

    def __init__(self, callback):
        super().__init__()
        self.callback = callback

    @pyqtSlot()
    def run(self):
        logger.info('Starting websocket server.')
        self.server = listen('localhost', 8765, self.callback)
        self.server.serve_forever()

    def stopServer(self):
        logger.info('Stopping websocket server.')
        self.server.shutdown()

class BrowserDialog(QWidget):
    uploadFinished = pyqtSignal()

    def __init__(self, parent=None, *args, **kwargs):
        super().__init__(parent, *args, **kwargs)

        self.worker: WebsocketServer = None

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.label = QLabel(self, text=self.tr('Please follow the instructions in the browser.'))
        layout.addWidget(self.label)

        self.threadpool = QThreadPool()

    @pyqtSlot(str, int, Any)
    def createWebsocketServer(self, apiHost: str, project: Project):
        self.worker = WebsocketServer(lambda websocket: handleConnection(websocket, project, apiHost, self))
        self.threadpool.start(self.worker)

    def closeWebsocketServer(self):
        if self.worker:
            self.worker.stopServer()
        self.uploadFinished.emit()

class MainWindow(QWidget):
    def __init__(self, parent=None, projectPath: Path=None, host: str=None, uploadURL: str=None, no_update: bool=False, *args, **kwargs):
        super().__init__(parent, *args, **kwargs)

        self.projectPath = projectPath
        self.host = host
        self.uploadURL = uploadURL

        self.resize(600, 60)
        filePath = Path(__file__).parent / 'assets' / 'millhouse.png'
        self.setWindowIcon(QIcon(str(filePath)))

        layout = QVBoxLayout()
        self.setLayout(layout)

        self.stackedWidget = QStackedWidget(self)
        layout.addWidget(self.stackedWidget)
        
        self.updatingDialog = UpdatingDialog(self)
        self.stackedWidget.addWidget(self.updatingDialog)

        self.browserDialog = BrowserDialog(self)
        self.stackedWidget.addWidget(self.browserDialog)

        self.projectSelector = ProjectSelector(self)
        self.stackedWidget.addWidget(self.projectSelector)

        self.updatingDialog.restartSignal.connect(self.close)

        if self.projectPath:
            self.updatingDialog.noUpdateSignal.connect(self.showBrowserDialog)
            self.updatingDialog.noUpdateSignal.connect(self.startWebsocket)
        else:
            self.updatingDialog.noUpdateSignal.connect(self.showProjectSelector)
            self.projectSelector.canceledSelection.connect(self.close)
            self.projectSelector.folderSelected.connect(self.setProjectPath)

        self.browserDialog.uploadFinished.connect(self.close)

        if no_update:
            if self.projectPath:
                self.stackedWidget.setCurrentWidget(self.browserDialog)
            else:
                self.stackedWidget.setCurrentWidget(self.projectSelector)

            self.updatingDialog.noUpdateSignal.emit()
        else:
            self.stackedWidget.setCurrentWidget(self.updatingDialog)
            self.updatingDialog.initializeUpdate()

    @pyqtSlot()
    def showBrowserDialog(self):
        logger.info(f'Opening browser with URL: {self.uploadURL}.')
        webbrowser.open_new_tab(self.uploadURL)

        self.stackedWidget.setCurrentWidget(self.browserDialog)
        self.update()

    @pyqtSlot()
    def startWebsocket(self):
        self.project = Project(self.projectPath)
        self.browserDialog.createWebsocketServer(self.host, self.project)

    @pyqtSlot()
    def showProjectSelector(self):
        self.stackedWidget.setCurrentWidget(self.projectSelector)
        self.update()

    @pyqtSlot(Path)
    def setProjectPath(self, projectPath: Path):
        self.projectPath = projectPath
        self.showBrowserDialog()
        self.startWebsocket()

    def closeEvent(self, event):
        self.browserDialog.closeWebsocketServer()
        return super().closeEvent(event)

def main():
    logging.basicConfig(stream=sys.stdout, level=logging.DEBUG, format='%(asctime)s %(name)s %(levelname)s: %(message)s')

    parser = argparse.ArgumentParser(description='millConnect order creator.')
    parser.add_argument('projectPath', nargs='?', type=Path, default=None, help='The path to a project folder.')
    parser.add_argument('--host', nargs='?', type=str, default='http://mhcicddev01.millhouse.local:8000/v1', help='The URL of the api, default: http://mhcicddev01.millhouse.local:8000/v1')
    parser.add_argument('--uploadURL', nargs='?', type=str, default='http://mhcicddev01.millhouse.local/uploadOrder', help='The URL to open in the browser, default: http://mhcicddev01.millhouse.local/uploadOrder')
    parser.add_argument('--no-update', action='store_true', help='Do not check for updates.')
    args = parser.parse_args()

    app = QApplication(sys.argv)
    app.setOrganizationName('millhouse GmbH')
    app.setApplicationName('millCONNECT Client')

    if args.projectPath and args.projectPath.is_file():
        args.projectPath = args.projectPath.parent

    try:
        my_version = metadata.version('millconnect')
        app.setApplicationVersion(my_version)
        logger.info(f'Starting millConnect client {my_version}.')
    except metadata.PackageNotFoundError:
        logger.info('Running from source code. Disabled update check.')
        args.no_update = True

    translator = QTranslator()
    filePath = Path(__file__).parent / 'assets'
    if translator.load('de_DE', str(filePath)):
        app.installTranslator(translator)

    mainWindow = MainWindow(projectPath=args.projectPath, host=args.host, uploadURL=args.uploadURL, no_update=args.no_update)
    mainWindow.show()

    ret = app.exec()
    sys.exit(ret)

if __name__ == '__main__':
    main()



    